import {
  View,
  Text,
  SafeAreaView,
  ScrollView,
  Button,
  Image,
  TouchableOpacity,
  Pressable,
  TextInput,
  Alert,
  Platform,
  Linking,
} from "react-native";
import React, { useEffect, useLayoutEffect, useRef, useState } from "react";

import { Ionicons } from "@expo/vector-icons";
import { COLORS } from "../styles/colors";
import { TextInputMask } from "react-native-masked-text";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import {
  addDoc,
  collection,
  doc,
  getDoc,
  getDocs,
  onSnapshot,
  query,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db, storage } from "../../config/firebase";
import * as Haptics from "expo-haptics";
import moment from "moment";
import * as Location from "expo-location";
import * as ImagePicker from "expo-image-picker";
import {
  getDownloadURL,
  getStorage,
  ref,
  uploadBytes,
  uploadBytesResumable,
} from "firebase/storage";
import BackButton from "../components/Buttons/BackButton";
import {
  BUTTON_CLASS,
  INPUT_CLASS,
  INPUT_CLASS_BORDER_BOTTOM,
  UPLOAD_PARAMS,
} from "../styles/constants";
import { SENDGRID_API_KEY, SENDGRID_FROM } from "@env";
import axios from "axios";
import PrimaryButton from "../components/Buttons/PrimaryButton";
import {
  cleanString,
  imageCompressor,
  openSettings,
  removeSpaces,
} from "../utils/allFunctions";
import SmallPrimaryButton from "../components/Buttons/SmallPrimaryButton";
import Animated, { FadeIn, FadeOut } from "react-native-reanimated";
import TransparentLoader from "../components/Loaders/TransparentLoader";
import sendAdminNotif from "../utils/sendAdminNotif";
import { GOOGLE_MAP_API_GEOCODING } from "@env";
import MapView, { PROVIDER_GOOGLE, Marker, Circle } from "react-native-maps";
import { useTranslation } from "react-i18next";

export default function Find({ navigation, route }) {
  const [supplement, setSupplement] = useState("");
  const [street, setStreet] = useState("");
  const [licensePlate, setLicensePlate] = useState("");
  const [postalCode, setPostalCode] = useState("");
  const [city, setCity] = useState("");
  const [coords, setCoords] = useState(null);
  const [country, setCountry] = useState("");
  const [check, setCheck] = useState(false);
  const [image, setImage] = useState(null);
  const [images, setImages] = useState([]);

  const [photoIsLoading, setPhotoIsLoading] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const { t, i18n } = useTranslation();

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: i18n.t("find_screen.headerTitle"),
    });
  });

  async function getLocation() {
    let { status } = await Location.requestForegroundPermissionsAsync();
    if (status !== "granted") {
      Alert.alert(
        i18n.t("find_screen.locationAccessDeniedTitle"),
        i18n.t("find_screen.locationAccessDeniedMessage"),
        [
          {
            text: i18n.t("find_screen.openSettings"),
            onPress: () => Linking.openSettings(),
          },
          {
            text: i18n.t("find_screen.cancel"),
            style: "cancel",
          },
        ],
        {
          cancelable: true,
        }
      );

      return;
    }

    try {
      let geoReverse = await Location.getCurrentPositionAsync({
        accuracy: Location.Accuracy.Balanced,
      });
      let location = await Location.reverseGeocodeAsync(geoReverse.coords);
      setPostalCode(location[0].postalCode);
      setCity(location[0].city);
      setCountry(location[0].country);
      setStreet(location[0].name);
      const { longitude, latitude } = geoReverse.coords;
      setCoords({
        long: longitude,
        lat: latitude,
      });
    } catch (error) {
      Alert.alert(i18n.t("find_screen.locationFetchingError"), error.message);
    }
  }

  async function sendPushNotifToSearcher(searcherInfo) {
    const message = {
      to: searcherInfo.expoPushToken,
      sound: "default",
      title: i18n.t("find_screen.pushNotificationTitle"),
      body: i18n.t("find_screen.pushNotificationBody"),
    };

    await fetch("https://exp.host/--/api/v2/push/send", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Accept-encoding": "gzip, deflate",
        "Content-Type": "application/json",
      },
      body: JSON.stringify(message),
    });
  }

  function removeSpaces(string) {
    return string.replace(/\s/g, "");
  }

  const sendMailToSearcher = async (searcherInfo) => {
    await axios.post(
      "https://api.sendgrid.com/v3/mail/send",
      {
        personalizations: [
          {
            to: [
              {
                email: `${searcherInfo.email}`,
              },
            ],
            dynamic_template_data: {
              first_name:
                searcherInfo.firstName !== ""
                  ? searcherInfo.firstName + " " + searcherInfo.lastName
                  : i18n.t("find_screen.emailTemplateDynamicFirstName"),
            },
          },
        ],
        from: {
          email: `${SENDGRID_FROM}`,
        },
        template_id: removeSpaces("d-d8a99978650a4a6e9149b3cd7fd445de"),
      },
      {
        headers: {
          authorization: `Bearer ${SENDGRID_API_KEY}`,
          "Content-Type": "application/json",
        },
      }
    );
  };

  async function sendNotifToSearcher(searcherId) {
    const dataToAdd = {
      title: i18n.t("find_screen.pushNotificationTitle"),
      text: i18n.t("find_screen.pushNotificationBody"),
      userId: searcherId,
      isViewed: false,
      createdAt: moment().format(),
    };

    await addDoc(collection(db, "notifications"), dataToAdd);
  }

  async function validate() {
    setIsLoading(true);
    if (!licensePlate || !street || !city || !country || !postalCode) {
      Alert.alert(
        i18n.t("find_screen.missingFieldsTitle"),
        i18n.t("find_screen.missingFieldsMessage")
      );

      setIsLoading(false);
      return;
    }

    if (!images.length) {
      Alert.alert(
        i18n.t("find_screen.addPhotosTitle"),
        i18n.t("find_screen.addPhotosMessage")
      );

      setIsLoading(false);
      return;
    }

    if (images.length >= 8) {
      Alert.alert(
        i18n.t("find_screen.maxPhotosTitle"),
        i18n.t("find_screen.maxPhotosMessage")
      );
      setIsLoading(false);
      return;
    }
    try {
      const q = query(
        collection(db, "searchs"),
        where("plate", "==", cleanString(licensePlate.toUpperCase())),
        where("finderId", "==", null),
        where("status", "==", "progress")
      );
      const querySnapshot = await getDocs(q);
      const dataArr = [];
      if (querySnapshot.size > 0) {
        querySnapshot.forEach((res) => {
          dataArr.push({ id: res.id, ...res.data() });
        });
        const findedSearchDemand = dataArr[0];
        const docSnap = await getDoc(
          doc(db, "users", findedSearchDemand.userId)
        );
        if (docSnap.exists()) {
          sendPushNotifToSearcher({ id: docSnap.id, ...docSnap.data() });
          sendMailToSearcher({ id: docSnap.id, ...docSnap.data() });
          sendNotifToSearcher(docSnap.id);
        }
        const imagesLinkArray = await uploadPhotos(images);

        await updateDoc(doc(db, "searchs", findedSearchDemand.id), {
          address: {
            street: street + " " + supplement,
            city,
            country,
            postalCode,
            coords,
          },
          photos: imagesLinkArray,
          status: "ended",
          paymentStatus: "unpaid",
          finderId: auth.currentUser.uid,
          finderPaymentStatus: "unpaid",
          findedAt: moment().format(),
        });

        setCheck(true);
        sendAdminNotif(`Un utilisateur a trouvé un véhicule : ${licensePlate}`);
        setIsLoading(false);
      } else {
        createAnnonce();
        return;
      }
    } catch (error) {
      setIsLoading(false);
      Alert.alert(
        i18n.t("find_screen.announcementNotPublishedTitle"),
        `${error}`,
        [{ text: "Ok" }]
      );
      return;
    }
  }

  async function createAnnonce() {
    setIsLoading(true);

    try {
      const imagesLinkArray = await uploadPhotos(images);
      await addDoc(collection(db, "searchs"), {
        addedViaFindScreen: true,
        country: "",
        assurance: "",
        plate: cleanString(licensePlate),
        type: "",
        userId: null,
        finderId: auth?.currentUser.uid,
        status: "progress",
        price: "",
        address: {
          street,
          city,
          country,
          postalCode,
          coords,
        },
        photos: imagesLinkArray,
        finderPrice: "",
        paymentStatus: "unpaid",
        finderPaymentStatus: "unpaid",
        matchAnimation: false,
        createdAt: moment().format(),
        findedAt: moment().format(),
      });

      Alert.alert(
        i18n.t("find_screen.reportReceivedTitle"),
        i18n.t("find_screen.reportReceivedMessage"),
        [{ text: "OK", onPress: () => navigation.goBack() }],
        {
          cancelable: true,
        }
      );
      sendAdminNotif(
        `Véhicule retrouvé mais pas de chercheur : ${licensePlate}`
      );
      setIsLoading(false);
    } catch (error) {
      setIsLoading(false);
      Alert.alert("An error occurred", `${error}`);
      return;
    }
  }

  const pickImage = async () => {
    Alert.alert(
      i18n.t("find_screen.selectImageOptionTitle"),
      i18n.t("find_screen.selectImageOptionMessage"),
      [
        {
          text: i18n.t("find_screen.cancel"),
          style: "cancel",
        },
        {
          text: i18n.t("find_screen.camera"),
          onPress: async () => {
            const { status } =
              await ImagePicker.requestCameraPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchCameraAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            if (!result.canceled) {
              const compressedImage = await imageCompressor(
                result.assets[0].uri
              );
              setImages((current) => [...current, ...[compressedImage]]);
            }
          },
        },
        {
          text: i18n.t("find_screen.library"),
          onPress: async () => {
            const { status } =
              await ImagePicker.requestMediaLibraryPermissionsAsync();
            if (status !== "granted") {
              openSettings();
              return;
            }
            const result = await ImagePicker.launchImageLibraryAsync({
              mediaTypes: ImagePicker.MediaTypeOptions.Images,
              allowsMultipleSelection: true,
              aspect: UPLOAD_PARAMS.aspect,
              quality: UPLOAD_PARAMS.quality,
            });
            if (!result.canceled) {
              setPhotoIsLoading(true);
              const newImagesPromises = result.assets.map(async (asset) => {
                const compressedImage = await imageCompressor(asset.uri);
                return compressedImage;
              });

              const newImages = await Promise.all(newImagesPromises);

              setImages((current) => [...current, ...newImages]);
              setTimeout(() => {
                setPhotoIsLoading(false);
              }, 500);
            }
          },
        },
      ],
      { cancelable: true }
    );
  };

  const uploadPhotos = async (images) => {
    try {
      const imageUrls = [];

      for (const image of images) {
        const response = await fetch(image);
        const blobFile = await response.blob();

        const storageRef = ref(
          storage,
          `findedVehiclesPhotos/${licensePlate.toUpperCase()}` +
            "_" +
            Date.now()
        );

        const snapshot = await uploadBytesResumable(storageRef, blobFile);

        const imageUrl = await getDownloadURL(snapshot?.ref);
        imageUrls.push(imageUrl);
      }

      return imageUrls;
    } catch (error) {
      console.error(i18n.t("find_screen.uploadingImagesError"), error);
    }
  };

  if (check) {
    return (
      <View
        style={{ marginTop: "-50%" }}
        className="flex-1 items-center justify-center bg-white"
      >
        <Image
          className="w-24 h-24"
          source={require("../../assets/img/trophee.png")}
        />
        <View className="w-10/12 self-center">
          <Text
            className="mt-10 text-lg text-black text-center"
            style={{ fontFamily: "Inter_500Medium" }}
          >
            {i18n.t("find_screen.reportReceivedTitle")}
          </Text>
          <Text className="mt-3 mb-5 text-sm text-gray-700 text-center">
            {i18n.t("find_screen.reportReceivedMessage")}
          </Text>
        </View>
      </View>
    );
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      {photoIsLoading && <TransparentLoader />}
      <KeyboardAwareScrollView
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        resetScrollToCoords={{ x: 0, y: 0 }}
        scrollEnabled
        contentContainerStyle={{ paddingBottom: "100%" }}
      >
        <View className="p-4">
          <View>
            <View className="mb-3 flex-row items-center justify-between">
              <Text
                className="text-lg text-black w-1/2"
                style={{ fontFamily: "Inter_500Medium" }}
              >
                {t("find_screen.photos")}
              </Text>
            </View>
            <View className="">
              <TouchableOpacity
                activeOpacity={0.5}
                onPress={pickImage}
                style={{ borderColor: images.length >= 8 ? "red" : "gray" }}
                className={`border border-dashed bg-gray-100 py-7 px-3 rounded-md flex-row items-center justify-between`}
              >
                <Text
                  className="text-lg text-gray-500"
                  style={{ fontFamily: "Inter_400Regular" }}
                >
                  {t("find_screen.touch_to_add", { count: images.length })}
                </Text>
                <View className="w-10 h-10 items-center justify-center rounded-full bg-gray-200">
                  <Ionicons name="camera-outline" size={25} color={"black"} />
                </View>
              </TouchableOpacity>
              <View className="flex-row flex-wrap justify-between">
                {images.map((item, index) => (
                  <Animated.View
                    entering={FadeIn}
                    exiting={FadeOut}
                    key={index}
                    className="mt-4 relative w-[48%] h-[20vh]"
                  >
                    <Image
                      source={{ uri: item }}
                      resizeMode={"cover"}
                      className="w-full h-full shadow bg-white rounded-lg border border-gray-200"
                    />
                    <TouchableOpacity
                      onPress={() => {
                        const newData = [...images];
                        newData.splice(index, 1);
                        setImages(newData);
                      }}
                      className="bg-red-500 absolute top-2 right-2 w-7 h-7 rounded-full flex items-center justify-center"
                    >
                      <Ionicons name="close" size={25} color="white" />
                    </TouchableOpacity>
                  </Animated.View>
                ))}
              </View>
            </View>
          </View>
          <View className="mt-8">
            <Text
              className="text-lg text-black mb-1"
              style={{ fontFamily: "Inter_500Medium" }}
            >
              {t("find_screen.license_plate")}
            </Text>
            <Pressable
              className="mb-2"
              onPress={() =>
                Alert.alert(
                  "Information",
                  t("find_screen.why_license_plate_response")
                )
              }
            >
              <Text
                className="text-sm text-gray-600"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                {t("find_screen.why_license_plate")}{" "}
                <Text
                  className="text-sm text-blue-500 underline"
                  style={{ fontFamily: "Inter_400Regular" }}
                >
                  {t("find_screen.click_to_read")}
                </Text>
              </Text>
            </Pressable>
            <TextInput
              className={INPUT_CLASS_BORDER_BOTTOM}
              placeholder={t("find_screen.license_plate")}
              placeholderTextColor={"gray"}
              value={licensePlate}
              onChangeText={(val) => {
                setLicensePlate(val.toUpperCase());
              }}
            />
          </View>
          <View className="mt-8">
            <View className="mb-1">
              <Text
                className="text-lg text-black w-1/2"
                style={{ fontFamily: "Inter_500Medium" }}
              >
                {t("find_screen.location")}
              </Text>
              <Text
                className="text-sm text-gray-600"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                {t("find_screen.location_description")}
              </Text>
            </View>
            <View className="my-5">
              <PrimaryButton
                onPress={getLocation}
                text={t("find_screen.fill_address_automatically")}
              />
            </View>

            <TextInput
              className={INPUT_CLASS_BORDER_BOTTOM}
              placeholder={t("find_screen.street_number")}
              placeholderTextColor={"gray"}
              value={street}
              onChangeText={(val) => {
                setStreet(val);
              }}
            />
            <View className="mt-2 flex-row items-center justify-between">
              <TextInput
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder={t("find_screen.supplement")}
                placeholderTextColor={"gray"}
                value={supplement}
                onChangeText={(val) => {
                  setSupplement(val);
                }}
              />
            </View>
            <View className="mt-2 flex-row items-center justify-between">
              <TextInput
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder={t("find_screen.postal_code")}
                placeholderTextColor={"gray"}
                value={postalCode}
                keyboardType="phone-pad"
                onChangeText={(val) => {
                  setPostalCode(val);
                }}
              />
            </View>
            <View className="mt-2 flex-row items-center justify-between">
              <TextInput
                style={{ width: "49%" }}
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder={t("find_screen.city")}
                placeholderTextColor={"gray"}
                value={city}
                onChangeText={(val) => {
                  setCity(val);
                }}
              />
              <TextInput
                style={{ width: "49%" }}
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder={t("find_screen.country")}
                placeholderTextColor={"gray"}
                value={country}
                onChangeText={(val) => {
                  setCountry(val);
                }}
              />
            </View>
          </View>

          {Platform.OS === "ios" && coords && (
            <View className="h-[30vh] mt-8">
              <MapView
                pointerEvents="none"
                scrollEnabled={false}
                provider={PROVIDER_GOOGLE}
                style={{
                  width: "100%",
                  height: "100%",
                  borderRadius: 5,
                  borderColor: "lightgray",
                  borderWidth: 2,
                }}
                initialCamera={{
                  center: { latitude: coords?.lat, longitude: coords?.long },
                  pitch: 90,
                  altitude: 300,
                  heading: 0,
                  zoom: 14,
                }}
                camera={{
                  center: coords,
                  pitch: 90,
                  altitude: 300,
                  heading: 0,
                  zoom: 14,
                }}
              >
                <Marker
                  coordinate={{
                    latitude: coords?.lat,
                    longitude: coords?.long,
                  }}
                />
                <Circle
                  center={{ latitude: coords?.lat, longitude: coords?.long }}
                  radius={600}
                  strokeWidth={2}
                  strokeColor="rgba(0,0,255,0.5)"
                  fillColor="rgba(0,0,255,0.1)"
                />
              </MapView>
            </View>
          )}

          <View className="mt-5 w-full bg-white">
            <PrimaryButton
              text={t("find_screen.send")}
              isLoading={isLoading}
              disabled={isLoading}
              onPress={validate}
            />
          </View>
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
