import {
  Text,
  SafeAreaView,
  TouchableOpacity,
  Alert,
  Image,
  TextInput,
  Keyboard,
} from "react-native";
import React, { useLayoutEffect, useState } from "react";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS_BORDER_BOTTOM,
} from "../../styles/constants";
import { FlashList } from "@shopify/flash-list";
import { View } from "react-native";
import PageLoader from "../../components/Loaders/PageLoader";
import { COLORS } from "../../styles/colors";
import { removeSpaces } from "../../utils/allFunctions";
import { COUNTRIES } from "../../utils/countries";
import { useTranslation } from "react-i18next";
import Animated, {
  FadeInRight,
  FadeOut,
} from "react-native-reanimated";
import BackButton from "../../components/Buttons/BackButton";

const SearchStep1 = ({ navigation, route }) => {
  const { userInfo } = route.params;
  const { t } = useTranslation();
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isLoading, setIsLoading] = useState(false);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("text.chooseCountry"),
    });
  }, [navigation, t]);

  async function validate(item) {
    if (!item) {
      Alert.alert(t("text.chooseCountry"));
      return;
    }
    navigation.navigate("SearchStep2", { selectedCountry: item, userInfo });
  }

  const filterCountries = (text) => {
    if (text === "") {
      return COUNTRIES;
    }
    const filteredCountries = COUNTRIES.filter((country) =>
      country.name.toLowerCase().startsWith(removeSpaces(text.toLowerCase()))
    );
    return filteredCountries;
  };

  // Rendu de la liste filtrée
  const renderFilteredCountries = () => {
    const filteredCountries = filterCountries(searchTerm);

    return (
      <>
        <View className="pt-4 px-4 mb-4 flex-row items-center justify-between w-full">
          <TextInput
            style={{ fontFamily: "Inter_400Regular" }}
            className={`${INPUT_CLASS_BORDER_BOTTOM} rounded-xl border pb-3 flex-1`}
            placeholder={`${t("text.chooseCountry")} (${COUNTRIES.length})`}
            placeholderTextColor="gray"
            onChangeText={(text) => setSearchTerm(text)}
            value={searchTerm}
          />
          {searchTerm && (
            <Animated.View
              className="w-3/12 rounded-md bg-red-500 p-2 ml-2 h-full"
              entering={FadeInRight}
              exiting={FadeOut}
            >
              <TouchableOpacity onPress={() => setSearchTerm("")}>
                <Text
                  style={{ fontFamily: "Inter_400Regular" }}
                  className="text-md text-white text-center mt-2"
                >
                  {t("search.clear")}
                </Text>
              </TouchableOpacity>
            </Animated.View>
          )}
        </View>
        <FlashList
          keyboardShouldPersistTaps="handled"
          contentContainerStyle={{
            paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
          }}
          data={filteredCountries}
          renderItem={({ item }) => (
            <TouchableOpacity
              className="flex-row items-center p-4 border-b border-gray-200"
              onPress={() => {
                Keyboard?.dismiss();
                validate(item?.code);
              }}
            >
              <Image
                source={{ uri: item?.flagUrl }}
                className="mr-4 w-10 h-7 z-20"
              />
              <Text
                style={{ fontFamily: "Inter_400Regular" }}
                className="text-base"
              >
                {item?.name}
              </Text>
            </TouchableOpacity>
          )}
          keyExtractor={(item) => item.code.toString()}
          estimatedItemSize={180}
          ListEmptyComponent={() => (
            <View style={{ padding: 10 }}>
              <Text
                style={{ fontFamily: "Inter_400Regular" }}
                className="text-base"
              >
                {t("search.noCountryFound")}
              </Text>
            </View>
          )}
        />
      </>
    );
  };

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      {renderFilteredCountries()}
    </SafeAreaView>
  );
};
export default SearchStep1;
