import {
  SafeAreaView,
  Alert,
  TextInput,
  View,
} from "react-native";
import React, { useLayoutEffect, useState } from "react";
import {
  addDoc,
  collection,
  doc,
  getDocs,
  query,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS_BORDER_BOTTOM,
} from "../../styles/constants";
import moment from "moment";
import PageLoader from "../../components/Loaders/PageLoader";
import { COLORS } from "../../styles/colors";
import { cleanString } from "../../utils/allFunctions";
import sendAdminNotif from "../../utils/sendAdminNotif";
import { useNavigation } from "@react-navigation/native";
import { useTranslation } from "react-i18next";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { SENDGRID_API_KEY, SENDGRID_FROM } from "@env";
import axios from "axios";
import BackButton from "../../components/Buttons/BackButton";
import PrimaryButton from "../../components/Buttons/PrimaryButton";

const SearchStep3 = ({ route }) => {
  const { selectedCountry, cat, userInfo } = route.params;
  const { t } = useTranslation();

  const [vehicleId, setVehicleId] = useState("");
  const [assurance, setAssurance] = useState("");
  const [isLoading, setIsLoading] = useState(false);

  const navigation = useNavigation();

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("additionalInfo.headerTitle"),
    });
  }, [navigation, t]);

  async function sendPushNotifToSearcher() {
    const message = {
      to: userInfo.expoPushToken,
      sound: "default",
      title: "Vous avez un match ! ✅",
      body: "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule... RDV dans l'onglet 'Démarches'",
    };

    await fetch("https://exp.host/--/api/v2/push/send", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Accept-encoding": "gzip, deflate",
        "Content-Type": "application/json",
      },
      body: JSON.stringify(message),
    });
  }

  const sendMailToSearcher = async () => {
    try {
      const response = await axios.post(
        "https://api.sendgrid.com/v3/mail/send",
        {
          personalizations: [
            {
              to: [
                {
                  email: `${auth?.currentUser?.email}`,
                },
              ],
              dynamic_template_data: {
                first_name: auth?.currentUser?.displayName,
              },
            },
          ],
          from: {
            email: `${SENDGRID_FROM}`,
          },
          template_id: cleanString("d-d8a99978650a4a6e9149b3cd7fd445de"),
        },
        {
          headers: {
            authorization: `Bearer ${SENDGRID_API_KEY}`,
            "Content-Type": "application/json",
          },
        }
      );

      console.log("Email sent successfully:", response.data);
    } catch (error) {
      console.error("Error sending email:", error);
    }
  };

  async function sendNotifToSearcher() {
    const dataToAdd = {
      title: "Vous avez un match ! ✅",
      title_en: "You have a match! ✅",
      text: "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule... RDV dans l'onglet 'Démarches'",
      text_en: "Your report within the Sur4plots application has enabled us to find your vehicle... Go to the 'Procedures' tab",
      userId: auth?.currentUser?.uid,
      isViewed: false,
      createdAt: moment().format(),
    };

    await addDoc(collection(db, "notifications"), dataToAdd);
  }

  async function validate() {
    setIsLoading(true);
    if (!vehicleId) {
      Alert.alert(
        t("additionalInfo.plateNumber")
      );
      setIsLoading(false);
      return;
    }

    if (!assurance) {
      Alert.alert(t("additionalInfo.insurance"));
      setIsLoading(false);
      return;
    }

    try {
      setIsLoading(true);

      const q = query(
        collection(db, "searchs"),
        where("plate", "==", cleanString(vehicleId.toUpperCase())),
        where("userId", "==", null),
        where("status", "==", "progress")
      );
      const querySnapshot = await getDocs(q);
      const dataArr = [];
      if (querySnapshot.size > 0) {
        querySnapshot.forEach((res) => {
          dataArr.push({ id: res.id, ...res.data() });
        });
        const findedSearchDemand = dataArr[0];

        await updateDoc(doc(db, "searchs", findedSearchDemand.id), {
          country: selectedCountry,
          type: cat,
          assurance: assurance,
          status: "ended",
          userId: auth?.currentUser.uid,
          finderPaymentStatus: "unpaid",
          createdAt: moment().format(),
        });

        Alert.alert(
          t("additionalInfo.vehicleFound"),
          t("additionalInfo.vehicleFoundMessage"),
          [
            {
              text: "J'ai compris",
              onPress: () => navigation.popToTop(),
            },
          ],
          {
            cancelable: true,
          }
        );
        sendPushNotifToSearcher();
        sendMailToSearcher();
        sendNotifToSearcher();
        sendAdminNotif(`Un chercheur a trouvé son véhicule: ${vehicleId}`);
      } else {
        await addDoc(collection(db, "searchs"), {
          country: selectedCountry,
          assurance: assurance,
          plate: cleanString(vehicleId),
          type: cat,
          userId: auth?.currentUser.uid,
          finderId: null,
          status: "progress",
          price: "",
          finderPrice: "",
          paymentStatus: "unpaid",
          finderPaymentStatus: "unpaid",
          matchAnimation: false,
          createdAt: moment().format(),
        });

        Alert.alert(
          t("additionalInfo.searchAcknowledged"),
          t("additionalInfo.searchAcknowledgedMessage"),
          [
            {
              text: "OK",
              onPress: () => navigation.popToTop(),
            },
          ],
          {
            cancelable: true,
          }
        );
        sendAdminNotif(`Nouvelle recherche de véhicule: ${vehicleId}`);
      }
    } catch (error) {
      Alert.alert(t("additionalInfo.errorOccurred"), `${error}`);
      return;
    } finally {
      setIsLoading(false);
    }
  }

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        className="px-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <TextInput
          className={INPUT_CLASS_BORDER_BOTTOM}
          style={{
            fontFamily: "Inter_400Regular",
            marginTop: 20,
            color: COLORS.black,
          }}
          placeholder={t("additionalInfo.plateNumber")}
          placeholderTextColor={"rgba(87, 96, 111,1.0)"}
          value={vehicleId}
          onChangeText={(text) => setVehicleId(text.toUpperCase())}
        />

        <TextInput
          className={INPUT_CLASS_BORDER_BOTTOM}
          style={{
            fontFamily: "Inter_400Regular",
            marginTop: 10,
            color: COLORS.black,
          }}
          placeholder={t("additionalInfo.insurance")}
          placeholderTextColor={"rgba(87, 96, 111,1.0)"}
          value={assurance}
          onChangeText={setAssurance}
        />

        {/* login button */}
        <View className="mt-10 w-full bg-white flex-1">
          <PrimaryButton
            text={t("buttons.search")}
            isLoading={isLoading}
            onPress={validate}
          />
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
};

export default SearchStep3;
