import { View, Text, SafeAreaView } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import * as Haptics from "expo-haptics";
import BackButton from "../../components/Buttons/BackButton";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS,
} from "../../styles/constants";
import PrimaryButton from "../../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { TextInput, Alert } from "react-native";
import { doc, updateDoc } from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import { updateProfile } from "firebase/auth";
import { useTranslation } from "react-i18next";

export default function PersonalInformations({ navigation, route }) {
  const { t } = useTranslation(); // Accès aux traductions
  const { data } = route.params;
  const [isLoading, setIsLoading] = useState(false);
  const [firstName, setFirstName] = useState(data?.firstName);
  const [lastName, setLastName] = useState(data?.lastName);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("personal_info_header"),
    });
  }, [navigation, t]);

  useEffect(() => {
    navigation
      .getParent()
      ?.setOptions({ tabBarStyle: { display: "none" }, tabBarVisible: false });
    return () =>
      navigation
        .getParent()
        ?.setOptions({ tabBarStyle: undefined, tabBarVisible: undefined });
  }, [navigation]);

  const validate = async () => {
    try {
      await updateDoc(doc(db, "users", auth.currentUser?.uid), {
        firstName: firstName,
        lastName: lastName,
      });
      await updateProfile(auth.currentUser, {
        displayName: firstName + " " + lastName,
      });

      Alert.alert(t("success_title"), t("success_message"), [
        { text: t("ok_button"), onPress: () => navigation.popToTop() },
      ]);
    } catch (error) {
      Alert.alert(t("error_title"), `${error.message}`);
    }
  };

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <KeyboardAwareScrollView
        className="p-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View>
          <View className="w-full">
            <TextInput
              style={{ fontFamily: "Inter_500Medium" }}
              className={INPUT_CLASS}
              placeholder={t("last_name_placeholder")}
              placeholderTextColor={"gray"}
              value={lastName}
              onChangeText={(val) => {
                setLastName(val);
              }}
            />
          </View>
          <View className="mt-3 w-full">
            <TextInput
              style={{ fontFamily: "Inter_500Medium" }}
              className={INPUT_CLASS}
              placeholder={t("first_name_placeholder")}
              placeholderTextColor={"gray"}
              value={firstName}
              onChangeText={(val) => {
                setFirstName(val);
              }}
            />
          </View>
          <View className="mt-3">
            <TextInput
              editable={false}
              style={{
                fontFamily: "Inter_500Medium",
                backgroundColor: "rgba(0,0,0,0.1)",
              }}
              className={INPUT_CLASS}
              placeholderTextColor={"gray"}
              value={data?.email}
            />
          </View>
        </View>
        <View className="flex-1 mt-10 w-full">
          <View className="w-full">
            <PrimaryButton
              text={t("submit_button")}
              isLoading={isLoading}
              onPress={validate}
            />
          </View>
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
