import {
  View,
  Text,
  Linking,
  Alert,
  Pressable,
  Share,
  Platform,
  AppState,
  StatusBar,
  RefreshControl,
  ScrollView,
  Image,
  ImageBackground,
  TouchableOpacity,
} from "react-native";
import React, {
  useCallback,
  useEffect,
  useLayoutEffect,
  useRef,
  useState,
} from "react";
import { addDoc, collection, doc, getDoc, updateDoc } from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import { sendEmailVerification, signOut } from "firebase/auth";
import moment from "moment/moment";
import PageSubTitle from "../../components/Titles/PageSubTitle";
import SettingsCard from "../../components/Settings/Card";
import PageLoader from "../../components/Loaders/PageLoader";
import * as Constants from "expo-constants";
import AsyncStorage from "@react-native-async-storage/async-storage";
import { COLORS } from "../../styles/colors";
import { useFocusEffect } from "@react-navigation/native";
import { Ionicons } from "@expo/vector-icons";
import Animated, { FadeIn, FadeOut } from "react-native-reanimated";
import { DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING } from "../../styles/constants";
import ProfileTopCard from "../../components/ProfileTopCard";
import BackButton from "../../components/Buttons/BackButton";
import { useTranslation } from "react-i18next";

export default function Profile({ navigation }) {
  const { t } = useTranslation();

  const scrollViewRef = useRef(null);
  const PADDING = "px-4 py-2";
  const buildNumber =
    Platform.OS === "ios"
      ? Constants.default.expoConfig.ios.buildNumber
      : Constants.default.expoConfig.android.versionCode;
  const [userInfo, setUserInfo] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [appState, setAppState] = React.useState(AppState.currentState);

  const appStoreURL = "https://apps.apple.com/us/app/id1641274230";

  const playStoreURL =
    "https://play.google.com/store/apps/details?id=com.sur4plot";
  const version = Constants.default.expoConfig.version;
  const website = "https://sur4plots.com/";

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: t("screens.profile.pageTitle"),
    });
  }, [navigation]);

  useFocusEffect(
    useCallback(() => {
      StatusBar.setBarStyle("dark-content");
    }, [])
  );

  useEffect(() => {
    getUserInfo();
  }, []);

  useFocusEffect(
    useCallback(() => {
      getUserInfo();
    }, [])
  );

  const topRefresh = () => {
    setIsLoading(true);
    auth?.currentUser?.reload();
    getUserInfo();
  };

  useEffect(() => {
    const subscription = AppState.addEventListener("change", (nextAppState) => {
      if (appState !== nextAppState) {
        setAppState(nextAppState);
        if (nextAppState == "active") {
          auth?.currentUser?.reload();
        }
      }
    });

    return () => {
      subscription.remove();
    };
  }, [appState]);

  const getUserInfo = async () => {
    if (!auth.currentUser) {
      setIsLoading(false);
      return;
    }
    try {
      const docRef = doc(db, "users", auth.currentUser.uid);
      const docSnap = await getDoc(docRef);

      if (docSnap.exists()) {
        setUserInfo({ id: docSnap.id, ...docSnap.data() });
      } else {
        Alert.alert(t("error_occurred"), t("data_retrieval_error"));
        return;
      }
    } catch (error) {
      Alert.alert(
        t("error_occurred"),
        t("data_retrieval_error") + `: ${error}`
      );
      return;
    } finally {
      setIsLoading(false);
    }
  };

  const shareData = async () => {
    await Share.share({
      message: t("share_message", { website }),
      url: website,
    });
  };

  const validateSignOut = () => {
    Alert.alert(
      t("logout_prompt"),
      t("logout_confirmation"),
      [
        { text: t("logout"), onPress: () => logOut() },
        {
          text: t("cancel"),
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  const logOut = async () => {
    setIsLoading(true);
    setTimeout(async () => {
      await AsyncStorage.removeItem("likedArticles");
      await signOut(auth);
      // navigation.popToTop();
    }, 500);
  };

  const onRefresh = async () => {
    setIsRefreshing(true);
    await getUserInfo();
    setIsRefreshing(false);
  };

  const validateUserDeleteAccount = () => {
    Alert.alert(
      t("account_closure_request"),
      t("account_closure_confirmation"),
      [
        { text: t("close_account"), onPress: () => userDeleteAccount() },
        {
          text: t("cancel"),
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  const validateSendEmailValidation = () => {
    Alert.alert(
      t("did_not_receive_verification_email", {
        email: auth.currentUser.email,
      }),
      t("resend_verification_email"),
      [
        { text: t("resend_link"), onPress: () => sendVerificationEmail() },
        {
          text: t("cancel"),
          style: "cancel",
        },
      ],
      {
        cancelable: true,
      }
    );
  };

  function sendVerificationEmail() {
    sendEmailVerification(auth.currentUser)
      .then(() => {
        Alert.alert(t("verification_email_sent"));
      })
      .catch((error) => {
        if (error.code == "auth/too-many-requests") {
          Alert.alert(t("error_occurred"), t("too_many_requests_error"));
          return;
        }
        Alert.alert(t("error_occurred"), `${error}`);
      });
  }

  const becameHost = async () => {
    setIsLoading(true);
    await updateDoc(doc(db, "users", auth.currentUser?.uid), {
      "host.isHost": true,
    });
    getUserInfo();
  };

  const userDeleteAccount = async () => {
    setIsLoading(true);
    await addDoc(collection(db, "suppressAccountDemands"), {
      userId: auth.currentUser.uid,
      isCompleted: false,
      createdAt: moment().format(),
    })
      .then(() => {
        Alert.alert(t("success_request_sent"), t("will_process_soon"));
      })
      .catch(() => {
        Alert.alert(t("oops_error"), t("error_occurred"));
      });
    setIsLoading(false);
  };

  const Card = ({ number, title, onPress }) => (
    <Pressable
      onPress={onPress}
      className="w-[45%] p-5 mx-1 mb-4 rounded-xl bg-gray-100"
    >
      <Text
        className="text-2xl text-black text-left"
        style={{ fontFamily: "Inter_700Bold" }}
      >
        {number}
      </Text>
      <Text className="text-sm text-gray-500">{title}</Text>
    </Pressable>
  );

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <View className="bg-white">
      <ScrollView
        bounces={false}
        refreshControl={
          <RefreshControl
            refreshing={isRefreshing}
            onRefresh={onRefresh}
            tintColor={COLORS.primary}
          />
        }
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING / 2,
        }}
      >
        {/* first section */}
        <ProfileTopCard
          getUserInfo={getUserInfo}
          userInfo={auth.currentUser && userInfo}
        />

        {auth.currentUser && !auth?.currentUser?.emailVerified && (
          <Animated.View entering={FadeIn} exiting={FadeOut}>
            <Pressable
              onPress={validateSendEmailValidation}
              className="mx-4 border border-yellow-500 bg-yellow-50 rounded-md mb-3 p-2 flex-row items-center justify-between"
            >
              <View className="flex-row items-center gap-2">
                <Image
                  style={{
                    height: 50,
                    width: 50,
                  }}
                  source={require("../../../assets/img/email.png")}
                  resizeMode={"cover"}
                />
                <Text
                  className="flex-1 text-sm text-yellow-700"
                  style={{ fontFamily: "Inter_300Light" }}
                >
                  {t("screens.profile.account.email1")}{" "}
                  <Text
                    className="flex-1 text-sm text-yellow-700"
                    style={{ fontFamily: "Inter_400Regular" }}
                  >
                    ({auth?.currentUser?.email})
                  </Text>{" "}
                  {t("screens.profile.account.email2")} {"\n"}
                  <Text
                    className="flex-1 text-sm text-yellow-700 underline"
                    style={{ fontFamily: "Inter_500Medium" }}
                  >
                    {t("screens.profile.account.email3")}
                  </Text>
                </Text>
              </View>
            </Pressable>
          </Animated.View>
        )}

        {!isLoading &&
          auth?.currentUser &&
          !userInfo?.verification?.isVerified && (
            <View className={PADDING}>
              <PageSubTitle title={`Vérification`} />
              <SettingsCard
                green
                title={t("screens.profile.account.verifyDoc")}
                iconName={"checkmark-outline"}
                onPress={() =>
                  navigation.navigate("ValidateProfileStep1", { userInfo })
                }
              />
            </View>
          )}
        {/* second section */}

        {/* paramètre */}
        {auth?.currentUser && (
          <View className={PADDING}>
            <PageSubTitle title={t("screens.profile.account.title")} />

            <SettingsCard
              title={t("screens.profile.account.personalInformations")}
              iconName={"person-outline"}
              onPress={() =>
                navigation.navigate("PersonalInformations", {
                  data: userInfo,
                })
              }
            />

            <SettingsCard
              title={t("screens.profile.account.SendRib")}
              iconName={"card-outline"}
              onPress={() =>
                navigation.navigate("SendRib", {
                  data: userInfo,
                })
              }
            />
          </View>
        )}

        <Pressable onPress={() => navigation.navigate("Tips")}>
          <ImageBackground
            resizeMode="cover"
            className="border border-gray-200 rounded-xl p-3 m-5"
            source={require("../../../assets/img/gbbb.png")}
            imageStyle={{ borderRadius: 10 }}
          >
            <View className="flex-row items-center">
              <Image
                resizeMode="contain"
                source={require("../../../assets/img/info.png")}
                className="w-32 h-32 mr-5"
              />
              <Text
                style={{ fontFamily: "Inter_500Medium" }}
                className="text-xl text-white"
              >
                {t("text.conseils")}
              </Text>
            </View>
          </ImageBackground>
        </Pressable>

        {/* Traduction */}
        <View className={PADDING}>
          <PageSubTitle title={t("screens.profile.traduction.title")} />

          <SettingsCard
            title={t("screens.profile.traduction.ChangeLanguage")}
            iconName={"language-outline"}
            onPress={() => navigation.navigate("ChangeLanguage")}
          />
        </View>

        {/* Commu */}
        <View className={PADDING}>
          <PageSubTitle title={t("screens.profile.commu.title")} />

          {/* <SettingsCard
            title={t("screens.profile.commu.contact")}
            iconName={"mail-outline"}
            onPress={() => Linking.openURL(`https://www.sur4plots.com/contact`)}
          /> */}

          <SettingsCard
            title={t("screens.profile.commu.share")}
            iconName={"share-outline"}
            onPress={shareData}
          />

          <SettingsCard
            title={`${t("screens.profile.commu.contact")}`}
            subTitle={`${t("screens.profile.commu.WriteComment")} ou ${t("screens.profile.support.problem")}`}
            iconName={"mail-outline"}
            // onPress={() => navigation.navigate("WriteComment")}

            onPress={() => Linking.openURL(`https://www.sur4plots.com/contact`)}
          />
        </View>

        {/* support */}
        <View className={PADDING}>
          <PageSubTitle title={t("screens.profile.support.title")} />

          <SettingsCard
            title={t("screens.profile.support.website")}
            iconName={"globe-outline"}
            onPress={() => Linking.openURL(`https://www.sur4plots.com/`)}
          />

          {/* <SettingsCard
            title={t("screens.profile.support.problem")}
            iconName={"bug-outline"}
            // onPress={() => navigation.navigate("SignalBug")}

            onPress={() => Linking.openURL(`https://www.sur4plots.com/contact`)}
          /> */}
        </View>
        {/* legal */}
        <View className={PADDING}>
          <PageSubTitle title={t("screens.profile.legal.title")} />

          <SettingsCard
            title={t("screens.profile.legal.policy")}
            iconName={"newspaper-outline"}
            onPress={() =>
              Linking.openURL(
                "https://www.sur4plots.com/politique-de-confidentialite/"
              )
            }
          />
        </View>

        {/* Autres */}
        {auth.currentUser && (
          <View className={PADDING}>
            <PageSubTitle title={t("screens.profile.others.title")} />

            <SettingsCard
              title={t("screens.profile.others.logout")}
              iconName={"log-out-outline"}
              onPress={validateSignOut}
            />
            <SettingsCard
              red
              title={t("screens.profile.others.delete")}
              iconName={"trash-outline"}
              onPress={validateUserDeleteAccount}
            />
          </View>
        )}
        <Text
          style={{ fontFamily: "Inter_400Regular" }}
          className="mt-10 text-center text-base text-gray-400"
        >
          Sur4Plots
        </Text>
        <Text
          style={{ fontFamily: "Inter_400Regular" }}
          className="text-center text-base text-gray-400"
        >
          {Constants.default.expoConfig.version} ({buildNumber})
        </Text>
      </ScrollView>
    </View>
  );
}
