import { View, Text, SafeAreaView } from "react-native";
import React, { useEffect, useLayoutEffect, useState } from "react";
import * as Haptics from "expo-haptics";
import BackButton from "../../../components/Buttons/BackButton";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS,
  INPUT_CLASS_BORDER_BOTTOM,
  INPUT_CLASS_STYLE_OPEN,
} from "../../../styles/constants";
import PrimaryButton from "../../../components/Buttons/PrimaryButton";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { TextInput, Alert } from "react-native";
import { doc, updateDoc } from "firebase/firestore";
import { auth, db } from "../../../../config/firebase";
import { formatNumber, removeSpaces } from "../../../utils/allFunctions";
import { updateProfile } from "firebase/auth";
import sendAdminNotif from "../../../utils/sendAdminNotif";

export default function SendRib({ navigation, route }) {
  const { data } = route.params;
  const [isLoading, setIsLoading] = useState(false);
  const [iban, setIban] = useState(data?.bankInfo?.iban);
  const [bic, setBic] = useState(data?.bankInfo?.bic);
  const [bankName, setBankName] = useState(data?.bankInfo?.bankName);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: "Coordonnées bancaires",
    });
  }, [navigation]);

  useEffect(() => {
    navigation
      .getParent()
      ?.setOptions({ tabBarStyle: { display: "none" }, tabBarVisible: false });
    return () =>
      navigation
        .getParent()
        ?.setOptions({ tabBarStyle: undefined, tabBarVisible: undefined });
  }, [navigation]);

  const validate = async () => {
    try {
      // Update the user document in Firestore
      await updateDoc(doc(db, "users", auth.currentUser?.uid), {
        bankInfo: {
          iban: removeSpaces(iban.toUpperCase()),
          bic: bic.toUpperCase(),
          bankName,
        },
      });

      Alert.alert(
        "Succès",
        "Vos informations bancaires ont été modifiées avec succès",

        [{ text: "Ok", onPress: () => navigation.goBack() }]
      );
      sendAdminNotif(
        `${
          data.firstName + " " + data.lastName
        } vient d'envoyer ses coordonnées bancaires`
      );
    } catch (error) {
      Alert.alert("Erreur lors de la mise à jour", `${error.message}`);
    }
  };

  return (
    <SafeAreaView className="flex-1 bg-gray-100">
      <KeyboardAwareScrollView
        className="p-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        // resetScrollToCoords={{ x: 0, y: 0 }}
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <View className="">
          <View className="">
            {/* flex view group for first and last name input */}
            <View className="w-full">
              {/* <Text
                className="text-sm mb-2 ml-1"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                IBAN
              </Text> */}
              <TextInput
                style={{ fontFamily: "Inter_500Medium" }}
                className={INPUT_CLASS}
                placeholder="IBAN *"
                maxLength={25}
                placeholderTextColor={"gray"}
                value={iban}
                onChangeText={(val) => {
                  setIban(formatNumber(val.toUpperCase(), 4));
                }}
              />
            </View>
            <View className="mt-3 w-full">
              {/* <Text
                className="text-sm mb-2 ml-1"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                Code BIC / SWIFT
              </Text> */}
              <TextInput
                style={{ fontFamily: "Inter_500Medium" }}
                className={INPUT_CLASS}
                placeholder="Code BIC / SWIFT *"
                placeholderTextColor={"gray"}
                value={bic}
                maxLength={12}
                onChangeText={(val) => {
                  setBic(val.toUpperCase());
                }}
              />
            </View>
            <View className="mt-3 w-full">
              {/* <Text
                className="text-sm mb-2 ml-1"
                style={{ fontFamily: "Inter_400Regular" }}
              >
                Nom de votre banque
              </Text> */}
              <TextInput
                style={{ fontFamily: "Inter_500Medium" }}
                className={INPUT_CLASS}
                placeholder="Nom de votre banque *"
                placeholderTextColor={"gray"}
                value={bankName}
                onChangeText={(val) => {
                  setBankName(val);
                }}
              />
            </View>
          </View>
          {/* validate button */}
          <View className="flex-1 mt-10 w-full">
            <View className="w-full">
              <PrimaryButton
                text={"Valider"}
                isLoading={isLoading}
                onPress={validate}
              />
            </View>
          </View>
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
